<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Apartment;
use Illuminate\Http\Request;

class ApartmentController extends Controller
{
    private function absoluteUrl(?string $path): ?string
    {
        if (!$path) return null;
        if (preg_match('#^https?://#i', $path)) {
            return $path;
        }
        return url($path);
    }

    private function transform($items)
    {
        return collect($items)->map(function ($a) {
            return [
                'id' => $a->id,
                'name' => $a->name,
                'description' => $a->description,
                'area' => $a->area,
                'price' => $a->price,
                'bedrooms' => $a->bedrooms,
                'bathrooms' => $a->bathrooms,
                'category' => $a->category,
                'location' => $a->location,
                'amenities' => $a->amenities,
                'status' => $a->status,
                'is_featured' => (bool) $a->is_featured,
                'image' => $this->absoluteUrl($a->image),
                'floor_plan' => $this->absoluteUrl($a->floor_plan),
                'created_at' => $a->created_at,
                'updated_at' => $a->updated_at,
            ];
        })->values();
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $apartments = Apartment::all();
        return response()->json(['data' => $this->transform($apartments)]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'required|string',
            'area' => 'required|string',
            'price' => 'required|string',
            'bedrooms' => 'required|integer',
            'bathrooms' => 'required|integer',
            'category' => 'required|string',
            'location' => 'required|string',
            'amenities' => 'nullable|array',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'gallery' => 'nullable|array',
            'floor_plan' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_featured' => 'nullable|boolean',
            'status' => 'nullable|in:available,sold,rented'
        ]);

        $data = $request->except(['image', 'floor_plan']);
        // Coerce boolean and handle amenities array
        $data['is_featured'] = $request->boolean('is_featured');

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('apartments', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        // Handle floor plan upload
        if ($request->hasFile('floor_plan')) {
            $floorPlanPath = $request->file('floor_plan')->store('apartments', 'public');
            $data['floor_plan'] = '/storage/' . $floorPlanPath;
        }

        $apartment = Apartment::create($data);
        return response()->json(['data' => $this->transform([$apartment])], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $apartment = Apartment::findOrFail($id);
        return response()->json(['data' => $this->transform([$apartment])->first()]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $apartment = Apartment::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'description' => 'sometimes|required|string',
            'area' => 'sometimes|required|string',
            'price' => 'sometimes|required|string',
            'bedrooms' => 'sometimes|required|integer',
            'bathrooms' => 'sometimes|required|integer',
            'category' => 'sometimes|required|string',
            'location' => 'sometimes|required|string',
            'amenities' => 'nullable|array',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'gallery' => 'nullable|array',
            'floor_plan' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_featured' => 'nullable|boolean',
            'status' => 'nullable|in:available,sold,rented'
        ]);

        $data = $request->except(['image', 'floor_plan']);
        if ($request->has('is_featured')) {
            $data['is_featured'] = $request->boolean('is_featured');
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('apartments', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        // Handle floor plan upload
        if ($request->hasFile('floor_plan')) {
            $floorPlanPath = $request->file('floor_plan')->store('apartments', 'public');
            $data['floor_plan'] = '/storage/' . $floorPlanPath;
        }

        $apartment->update($data);
        $apartment->refresh();
        return response()->json(['data' => $this->transform([$apartment])]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $apartment = Apartment::findOrFail($id);
        $apartment->delete();
        return response()->json(['success' => true]);
    }
}
