<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\BlogPost;
use Illuminate\Http\Request;

class BlogPostController extends Controller
{
    private function absoluteUrl(?string $path): ?string
    {
        if (!$path) return null;
        if (preg_match('#^https?://#i', $path)) {
            return $path;
        }
        return url($path);
    }

    private function transform($items)
    {
        return collect($items)->map(function ($b) {
            return [
                'id' => $b->id,
                'title' => $b->title,
                'content' => $b->content,
                'excerpt' => $b->excerpt,
                'author' => $b->author,
                'date' => $b->date,
                'is_featured' => (bool) $b->is_featured,
                'published' => (bool) $b->published,
                'image' => $this->absoluteUrl($b->image),
                'created_at' => $b->created_at,
                'updated_at' => $b->updated_at,
            ];
        })->values();
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $blogPosts = BlogPost::all();
        return response()->json(['data' => $this->transform($blogPosts)]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string',
            'author' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'date' => 'required|date',
            'is_featured' => 'nullable|boolean',
            'published' => 'nullable|boolean'
        ]);

        $data = $request->except(['image']);
        // Coerce boolean fields
        $data['is_featured'] = $request->boolean('is_featured');
        $data['published'] = $request->boolean('published');

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('blogs', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        $blogPost = BlogPost::create($data);
        return response()->json(['data' => $this->transform([$blogPost])], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $blogPost = BlogPost::findOrFail($id);
        return response()->json(['data' => $this->transform([$blogPost])->first()]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $blogPost = BlogPost::findOrFail($id);

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'content' => 'sometimes|required|string',
            'excerpt' => 'nullable|string',
            'author' => 'sometimes|required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'date' => 'sometimes|required|date',
            'is_featured' => 'nullable|boolean',
            'published' => 'nullable|boolean'
        ]);

        $data = $request->except(['image']);
        // Coerce boolean fields when present
        if ($request->has('is_featured')) {
            $data['is_featured'] = $request->boolean('is_featured');
        }
        if ($request->has('published')) {
            $data['published'] = $request->boolean('published');
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('blogs', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        $blogPost->update($data);
        $blogPost->refresh();
        return response()->json(['data' => $this->transform([$blogPost])]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $blogPost = BlogPost::findOrFail($id);
        $blogPost->delete();
        return response()->json(['success' => true]);
    }
}
