<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Booking;
use Illuminate\Http\Request;

class BookingController extends Controller
{
    // Public: accept booking submissions from the website
    public function store(Request $request)
    {
        // Backwards compatibility: accept legacy field names from existing frontend
        $payload = $request->all();
        $normalized = [
            'customer_name'   => $payload['customer_name']   ?? $payload['name']   ?? null,
            'customer_email'  => $payload['customer_email']  ?? $payload['email']  ?? null,
            'customer_phone'  => $payload['customer_phone']  ?? $payload['phone']  ?? null,
            'property_id'     => $payload['property_id']     ?? null,
            'check_in_date'   => $payload['check_in_date']   ?? $payload['visit_date'] ?? null,
            'check_out_date'  => $payload['check_out_date']  ?? null,
            'guests_count'    => $payload['guests_count']    ?? null,
            'special_requests'=> $payload['special_requests']?? $payload['message'] ?? null,
        ];
        // If a preferred time was sent in legacy form, append to special requests
        if (!empty($payload['visit_time'])) {
            $normalized['special_requests'] = trim(($normalized['special_requests'] ? $normalized['special_requests']."\n" : '') . 'Preferred time: ' . $payload['visit_time']);
        }

        // If no check_out_date provided, default to same as check_in_date
        if (empty($normalized['check_out_date'])) {
            $normalized['check_out_date'] = $normalized['check_in_date'];
        }

        $validated = validator($normalized, [
            'customer_name' => 'required|string|max:255',
            'customer_email' => 'required|email|max:255',
            'customer_phone' => 'nullable|string|max:100',
            'property_id' => 'nullable|exists:properties,id',
            'check_in_date' => 'required|date',
            'check_out_date' => 'nullable|date|after_or_equal:check_in_date',
            'guests_count' => 'nullable|integer|min:1',
            'special_requests' => 'nullable|string',
        ])->validate();

        $booking = Booking::create(array_merge($validated, ['status' => 'pending']));

        return response()->json([
            'success' => true,
            'data' => $booking->load('property'),
        ], 201);
    }

    // Protected: list all bookings for the dashboard
    public function index()
    {
        $bookings = Booking::with('property')->orderByDesc('created_at')->get();
        return response()->json([
            'success' => true,
            'data' => $bookings,
        ]);
    }

    // Update booking status
    public function update(Request $request, string $id)
    {
        $booking = Booking::findOrFail($id);

        $validated = $request->validate([
            'status' => 'required|in:pending,confirmed,cancelled,completed',
        ]);

        $booking->update($validated);

        return response()->json([
            'success' => true,
            'data' => $booking->load('property'),
        ]);
    }

    // Show specific booking
    public function show(string $id)
    {
        $booking = Booking::with('property')->findOrFail($id);
        return response()->json([
            'success' => true,
            'data' => $booking,
        ]);
    }
}
