<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\HeroSlide;
use Illuminate\Http\Request;

class HeroSlideController extends Controller
{
    private function absoluteUrl(?string $path): ?string
    {
        if (!$path) return null;
        if (preg_match('#^https?://#i', $path)) {
            return $path;
        }
        return url($path);
    }

    private function transform($slides)
    {
        return collect($slides)->map(function ($slide) {
            return [
                'id' => $slide->id,
                'title' => $slide->title,
                'subtitle' => $slide->subtitle,
                'order' => $slide->order,
                // Force all loaded slides to be active in API responses
                'active' => true,
                'background_image' => $this->absoluteUrl($slide->background_image),
                'interior_image' => $this->absoluteUrl($slide->interior_image),
                'created_at' => $slide->created_at,
                'updated_at' => $slide->updated_at,
            ];
        })->values();
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $heroSlides = HeroSlide::orderBy('order')->get();
        if (request()->boolean('active_only')) {
            $heroSlides = $heroSlides->where('is_active', true);
        }
        return response()->json(['data' => $this->transform($heroSlides)]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'background_image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'interior_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string',
            'order' => 'nullable|integer',
            'is_active' => 'nullable|boolean'
        ]);

        $data = $request->except(['background_image', 'interior_image']);
        // Enforce new slides are active regardless of client input
        $data['is_active'] = true;
        // Ensure required DB fields have safe defaults
        if (!array_key_exists('subtitle', $data) || $data['subtitle'] === null) {
            $data['subtitle'] = '';
        }
        if (!array_key_exists('order', $data) || $data['order'] === null || $data['order'] === '') {
            // place new slide at the end
            $max = (int) (HeroSlide::max('order') ?? 0);
            $data['order'] = $max + 1;
        }

        // Handle background image upload
        if ($request->hasFile('background_image')) {
            $bgImagePath = $request->file('background_image')->store('hero-slides', 'public');
            $data['background_image'] = '/storage/' . $bgImagePath;
        }

        // Handle interior image upload
        if ($request->hasFile('interior_image')) {
            $intImagePath = $request->file('interior_image')->store('hero-slides', 'public');
            $data['interior_image'] = '/storage/' . $intImagePath;
        }

        $heroSlide = HeroSlide::create($data);
        return response()->json(['data' => $this->transform([$heroSlide])], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $heroSlide = HeroSlide::findOrFail($id);
        return response()->json(['data' => $this->transform([$heroSlide])->first()]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $heroSlide = HeroSlide::findOrFail($id);

        $validated = $request->validate([
            'background_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'interior_image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'title' => 'sometimes|required|string|max:255',
            'subtitle' => 'sometimes|nullable|string',
            'order' => 'nullable|integer',
            'is_active' => 'nullable|boolean'
        ]);

        $data = $request->except(['background_image', 'interior_image']);
        // Coerce boolean field when present (support alias 'active')
        if ($request->has('is_active')) {
            $data['is_active'] = $request->boolean('is_active');
        } elseif ($request->has('active')) {
            $data['is_active'] = $request->boolean('active');
        }

        // Handle background image upload
        if ($request->hasFile('background_image')) {
            $bgImagePath = $request->file('background_image')->store('hero-slides', 'public');
            $data['background_image'] = '/storage/' . $bgImagePath;
        }

        // Handle interior image upload
        if ($request->hasFile('interior_image')) {
            $intImagePath = $request->file('interior_image')->store('hero-slides', 'public');
            $data['interior_image'] = '/storage/' . $intImagePath;
        }

        $heroSlide->update($data);
        $heroSlide->refresh();
        return response()->json(['data' => $this->transform([$heroSlide])]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $heroSlide = HeroSlide::findOrFail($id);
        $heroSlide->delete();
        return response()->json(['success' => true]);
    }
}
