<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class HousePlanController extends Controller
{
    public function generate(Request $request)
    {
        $validated = $request->validate([
            'prompt' => 'required|string|max:4000',
        ]);

        $prompt = $validated['prompt'];

        // Attempt to call OpenAI if an API key is configured
        $apiKey = env('OPENAI_API_KEY');
        if ($apiKey) {
            try {
                $payload = [
                    'model' => env('OPENAI_MODEL', 'gpt-4o-mini'),
                    'messages' => [
                        [
                            'role' => 'system',
                            'content' => 'You generate simple house plan JSON. Output ONLY valid JSON. Schema: {"plan": {"units": "meters", "rooms": [{"name": "Bedroom", "width": number, "height": number, "x": number, "y": number, "doorsTo": ["RoomName", ...]}]}}. Ensure non-overlapping rectangles. Fit within 12m x 12m by adjusting if needed.'
                        ],
                        [
                            'role' => 'user',
                            'content' => 'Create a house plan: ' . $prompt,
                        ],
                    ],
                    'temperature' => 0.3,
                ];

                $ch = curl_init('https://api.openai.com/v1/chat/completions');
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($ch, CURLOPT_HTTPHEADER, [
                    'Content-Type: application/json',
                    'Authorization: Bearer ' . $apiKey,
                ]);
                curl_setopt($ch, CURLOPT_POST, true);
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
                $resp = curl_exec($ch);
                $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
                $err = curl_error($ch);
                curl_close($ch);

                if ($err) {
                    throw new \RuntimeException('OpenAI request failed: ' . $err);
                }
                if ($http < 200 || $http >= 300) {
                    throw new \RuntimeException('OpenAI HTTP ' . $http . ' response: ' . $resp);
                }

                $data = json_decode($resp, true);
                $text = $data['choices'][0]['message']['content'] ?? '';

                // Extract JSON if the model wrapped it in code fences
                if (preg_match('/\{[\s\S]*\}/', $text, $m)) {
                    $text = $m[0];
                }
                $json = json_decode($text, true);

                if (json_last_error() === JSON_ERROR_NONE && isset($json['plan'])) {
                    return response()->json([
                        'success' => true,
                        'source' => 'openai',
                        'data' => $json,
                    ]);
                }
            } catch (\Throwable $e) {
                // Fall through to heuristic planner
            }
        }

        // Heuristic fallback: very simple layout grid based on keywords
        $lower = strtolower($prompt);
        $rooms = [];
        $wanted = [
            'living room' => 1,
            'kitchen' => 1,
            'bathroom' => 1,
        ];
        if (preg_match('/(\d+)\s*bed(room)?s?/i', $prompt, $m)) {
            $wanted['bedroom'] = (int) $m[1];
        } else {
            $wanted['bedroom'] = 2;
        }
        if (strpos($lower, 'garage') !== false) { $wanted['garage'] = 1; }

        foreach ($wanted as $name => $count) {
            for ($i = 1; $i <= $count; $i++) {
                $label = $count > 1 && $name !== 'bathroom' ? ucfirst($name) . ' ' . $i : ucfirst($name);
                $rooms[] = [
                    'name' => $label,
                    'width' => in_array($name, ['living room']) ? 5 : (in_array($name, ['kitchen']) ? 3.5 : 3),
                    'height' => in_array($name, ['living room']) ? 4 : 3,
                ];
            }
        }

        // Simple grid pack within 12x12
        $x = 0; $y = 0; $rowH = 0; $maxW = 12;
        foreach ($rooms as &$r) {
            $w = $r['width']; $h = $r['height'];
            if ($x + $w > $maxW) { $x = 0; $y += $rowH + 0.3; $rowH = 0; }
            $r['x'] = $x; $r['y'] = $y; $rowH = max($rowH, $h); $x += $w + 0.3;
            $r['doorsTo'] = [];
        }
        unset($r);

        $plan = [
            'plan' => [
                'units' => 'meters',
                'rooms' => $rooms,
            ],
        ];

        return response()->json([
            'success' => true,
            'source' => $apiKey ? 'fallback:openai_failed' : 'fallback:no_api_key',
            'data' => $plan,
        ]);
    }
}

