<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Property;
use Illuminate\Http\Request;

class PropertyController extends Controller
{
    private function absoluteUrl(?string $path): ?string
    {
        if (!$path) return null;
        // Already absolute
        if (preg_match('#^https?://#i', $path)) {
            return $path;
        }
        // If it already starts with a slash, keep it root-relative to avoid mixed content
        if (isset($path[0]) && $path[0] === '/') {
            return $path;
        }
        // Normalize common storage-disk relative paths (saved under public/storage)
        if (preg_match('#^(properties|apartments|hero-slides|blogs|testimonials)/#i', $path)) {
            return '/storage/' . ltrim($path, '/');
        }
        // Fallback to root-relative
        return '/' . ltrim($path, '/');
    }

    private function transform($items)
    {
        return collect($items)->map(function ($p) {
            $gallery = $p->gallery;
            if (is_array($gallery)) {
                $gallery = array_map(fn($g) => $this->absoluteUrl($g), $gallery);
            } elseif (is_string($gallery)) {
                $gallery = [$this->absoluteUrl($gallery)];
            }
            return [
                'id' => $p->id,
                'title' => $p->title,
                'description' => $p->description,
                'price' => $p->price,
                'location' => $p->location,
                'status' => $p->status,
                'is_featured' => (bool) $p->is_featured,
                'views_count' => (int) ($p->views_count ?? 0),
                'image' => $this->absoluteUrl($p->image),
                'gallery' => $gallery,
                'floor_plan' => $this->absoluteUrl($p->floor_plan),
                'created_at' => $p->created_at,
                'updated_at' => $p->updated_at,
            ];
        })->values();
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $properties = Property::all();
        return response()->json(['data' => $this->transform($properties)]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'nullable|numeric',
            'location' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'gallery' => 'nullable|array',
            'floor_plan' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'status' => 'nullable|in:available,sold,rented',
            'is_featured' => 'nullable|boolean',
            'views_count' => 'nullable|integer'
        ]);

        $data = $request->except(['image', 'floor_plan']);
        // Coerce checkbox/truthy input explicitly to boolean
        $data['is_featured'] = $request->boolean('is_featured');

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('properties', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        // Handle floor plan upload
        if ($request->hasFile('floor_plan')) {
            $floorPlanPath = $request->file('floor_plan')->store('properties', 'public');
            $data['floor_plan'] = '/storage/' . $floorPlanPath;
        }

        $property = Property::create($data);
        return response()->json(['data' => $this->transform([$property])], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $property = Property::findOrFail($id);
        return response()->json(['data' => $this->transform([$property])->first()]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $property = Property::findOrFail($id);

        $validated = $request->validate([
            'title' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'price' => 'nullable|numeric',
            'location' => 'nullable|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'gallery' => 'nullable|array',
            'floor_plan' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'status' => 'nullable|in:available,sold,rented',
            'is_featured' => 'nullable|boolean',
            'views_count' => 'nullable|integer'
        ]);

        $data = $request->except(['image', 'floor_plan']);
        // Only set when present; coerce to boolean
        if ($request->has('is_featured')) {
            $data['is_featured'] = $request->boolean('is_featured');
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('properties', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        // Handle floor plan upload
        if ($request->hasFile('floor_plan')) {
            $floorPlanPath = $request->file('floor_plan')->store('properties', 'public');
            $data['floor_plan'] = '/storage/' . $floorPlanPath;
        }

        $property->update($data);
        $property->refresh();
        return response()->json(['data' => $this->transform([$property])]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $property = Property::findOrFail($id);
        $property->delete();
        return response()->json(['success' => true]);
    }

    /**
     * Bulk import properties from CSV or JSON array.
     * Accepts either a multipart file 'file' (CSV with headers)
     * or a JSON payload 'items' which is an array of property objects.
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'nullable|file|mimes:csv,txt',
            'items' => 'nullable|array',
        ]);

        if (!$request->hasFile('file') && !$request->has('items')) {
            return response()->json([
                'success' => false,
                'message' => 'Provide a CSV file or an items array',
            ], 422);
        }

        $created = 0; $failed = 0; $errors = [];

        $upsert = function(array $row) use (&$created, &$failed, &$errors) {
            try {
                $data = [
                    'title' => trim((string)($row['title'] ?? '')),
                    'description' => $row['description'] ?? null,
                    'price' => $row['price'] ?? null,
                    'location' => $row['location'] ?? null,
                    'status' => in_array(($row['status'] ?? 'available'), ['available','sold','rented']) ? $row['status'] : 'available',
                    'is_featured' => filter_var(($row['is_featured'] ?? false), FILTER_VALIDATE_BOOLEAN),
                    'image' => $row['image'] ?? null,
                    'floor_plan' => $row['floor_plan'] ?? null,
                ];
                // gallery: comma/semicolon separated or array
                if (isset($row['gallery'])) {
                    if (is_array($row['gallery'])) {
                        $data['gallery'] = $row['gallery'];
                    } else {
                        $parts = preg_split('/[;,]/', (string)$row['gallery']);
                        $data['gallery'] = array_values(array_filter(array_map('trim', $parts), fn($v) => $v !== ''));
                    }
                }

                if ($data['title'] === '') {
                    throw new \InvalidArgumentException('Missing title');
                }

                Property::create($data);
                $created++;
            } catch (\Throwable $e) {
                $failed++;
                $errors[] = $e->getMessage();
            }
        };

        if ($request->has('items') && is_array($request->items)) {
            foreach ($request->items as $row) {
                if (is_array($row)) { $upsert($row); }
            }
        }

        if ($request->hasFile('file')) {
            if (($handle = fopen($request->file('file')->getRealPath(), 'r')) !== false) {
                $headers = null;
                while (($data = fgetcsv($handle, 0, ',')) !== false) {
                    if ($headers === null) {
                        $headers = array_map(function($h){ return strtolower(trim($h)); }, $data);
                        continue;
                    }
                    $row = [];
                    foreach ($headers as $i => $h) {
                        $row[$h] = $data[$i] ?? null;
                    }
                    $upsert($row);
                }
                fclose($handle);
            }
        }

        return response()->json([
            'success' => true,
            'summary' => [
                'created' => $created,
                'failed' => $failed,
            ],
            'errors' => $errors,
        ]);
    }
}
