<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Testimonial;
use Illuminate\Http\Request;

class TestimonialController extends Controller
{
    private function absoluteUrl(?string $path): ?string
    {
        if (!$path) return null;
        if (preg_match('#^https?://#i', $path)) {
            return $path;
        }
        return url($path);
    }

    private function transform($items)
    {
        return collect($items)->map(function ($t) {
            return [
                'id' => $t->id,
                'name' => $t->name,
                'role' => $t->role,
                'quote' => $t->quote,
                'is_featured' => (bool) $t->is_featured,
                'image' => $this->absoluteUrl($t->image),
                'created_at' => $t->created_at,
                'updated_at' => $t->updated_at,
            ];
        })->values();
    }
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $testimonials = Testimonial::all();
        return response()->json(['data' => $this->transform($testimonials)]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'role' => 'required|string|max:255',
            'quote' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_featured' => 'nullable|boolean'
        ]);

        $data = $request->except(['image']);
        $data['is_featured'] = $request->boolean('is_featured');

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('testimonials', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        $testimonial = Testimonial::create($data);
        return response()->json(['data' => $this->transform([$testimonial])], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $testimonial = Testimonial::findOrFail($id);
        return response()->json(['data' => $this->transform([$testimonial])->first()]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $testimonial = Testimonial::findOrFail($id);

        $validated = $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'role' => 'sometimes|required|string|max:255',
            'quote' => 'sometimes|required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'is_featured' => 'nullable|boolean'
        ]);

        $data = $request->except(['image']);
        if ($request->has('is_featured')) {
            $data['is_featured'] = $request->boolean('is_featured');
        }

        // Handle image upload
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('testimonials', 'public');
            $data['image'] = '/storage/' . $imagePath;
        }

        $testimonial->update($data);
        $testimonial->refresh();
        return response()->json(['data' => $this->transform([$testimonial])]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $testimonial = Testimonial::findOrFail($id);
        $testimonial->delete();
        return response()->json(['success' => true]);
    }
}
