<?php

use Illuminate\Support\Facades\Route;
use App\Http\Controllers\DashboardController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Laravel\Sanctum\PersonalAccessToken;

Route::get('/', function () {
    return view('welcome');
});

// Downloadable CSV template for property import
Route::get('/templates/properties-import.csv', function () {
    $csv = implode(",", [
        'title','description','price','location','status','is_featured','image','floor_plan','gallery'
    ])."\n".
    'Elegant Condo,"2 bed unit",250000,Downtown,available,true,https://example.com/image.jpg,,"https://img1.jpg; https://img2.jpg"' . "\n";

    return response($csv, 200, [
        'Content-Type' => 'text/csv',
        'Content-Disposition' => 'attachment; filename="properties-import-template.csv"'
    ]);
});

// Bridge: convert Sanctum API token to web session
// Option 1: Bearer token XHR (returns JSON)
Route::get('/session-login', function (Request $request) {
    Auth::login($request->user());
    $request->session()->regenerate();
    return response()->json(['success' => true]);
})->middleware('auth:sanctum');

// Option 2: Token in query with 302 redirect (no CORS needed)
Route::get('/session-login-redirect', function (Request $request) {
    $token = $request->query('token');
    if (!$token) {
        return redirect('/login');
    }
    $accessToken = PersonalAccessToken::findToken($token);
    if (!$accessToken) {
        return redirect('/login');
    }
    $user = $accessToken->tokenable;
    if (!$user) {
        return redirect('/login');
    }
    Auth::login($user);
    $request->session()->regenerate();
    return redirect('/dashboard');
});

// Dashboard routes (protected)
Route::middleware('auth')->group(function () {
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');
    Route::get('/dashboard/properties', [DashboardController::class, 'properties'])->name('dashboard.properties');
    Route::get('/dashboard/blogs', [DashboardController::class, 'blogs'])->name('dashboard.blogs');
    Route::get('/dashboard/testimonials', [DashboardController::class, 'testimonials'])->name('dashboard.testimonials');
    Route::get('/dashboard/hero-slides', [DashboardController::class, 'heroSlides'])->name('dashboard.hero-slides');
    Route::get('/dashboard/apartments', [DashboardController::class, 'apartments'])->name('dashboard.apartments');
    Route::get('/dashboard/bookings', [DashboardController::class, 'bookings'])->name('dashboard.bookings');

    // Logout route
    Route::post('/logout', function (Request $request) {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect('/login');
    })->name('logout');
});

// Authentication routes
Route::get('/login', function () {
    return view('auth.login');
})->name('login');

Route::get('/register', function () {
    return view('auth.register');
})->name('register');
